/**
 * URM – Process Monitoring
 *
 * Displays an administrative log of actions taken through the URM system.
 * Includes paginated viewing, export to CSV, and deletion functionality.
 * Log entries include timestamp, message, type (error, warning, info), and user ID.
 *
 * Author: Black & Copilot
 */

// 🔍 URM – Process Monitoring (admin log display)
			
function render_ur_process_log_tab() {	
	urm_guard_admin_view();
    echo '<div style="margin-top: 0px; background: #fff; border: 1px solid #ccc; border-radius: 0px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); overflow: hidden;">';

// 👑 Frame header
echo '<div style="background: #f7f9fa; padding: 10px 16px; border-bottom: 1px solid #ccc; font-weight: bold; font-size: 16px;">📊 Process monitoring</div>';

// 📦 Internal content
echo '<div style="padding: 0px 20px 20px 20px;">';

    $log = get_option('ur_process_log', []);

    if (empty($log)) {
        echo '<p style="color: gray;">📭 Currently, there are no logged messages.</p>';
        return;
    }

    // ✨ The admin can set the number of rows per page (or use the default value)
    $per_page = isset($_GET['per_page']) ? intval($_GET['per_page']) : 15;
    $per_page = ($per_page > 0 && $per_page <= 500) ? $per_page : 15;

    // ➕ Total number of entries per pages
    $total_items = count($log);
    $total_pages = ceil($total_items / $per_page);
    $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;

    //✂️ Cut the part of the log that is displayed on the current page.
    $log = array_reverse($log); // latest first
    $offset = ($current_page - 1) * $per_page;
    $log_slice = array_slice($log, $offset, $per_page);

    // 🧾 Table
    echo '<table class="widefat fixed striped" style="margin-top: 20px;">';
    echo '<thead><tr><th style="width:180px;">⏱️ Time</th><th>📝 Message</th><th style="width:100px;">🎯 Type</th><th style="width:80px;">👤 User</th></tr></thead>';
    echo '<tbody>';

    foreach ($log_slice as $entry) {
        $time = date('d.m.Y. H:i:s', $entry['timestamp']);
        $message = esc_html($entry['message']);
        $type = esc_html($entry['type']);
        $user = get_userdata($entry['user_id'] ?? 0);
        $user_display = $user ? esc_html($user->user_login) : '–';

        switch ($type) {
            case 'error': $color = '#dc3545'; $icon = '❌'; break;
            case 'warning': $color = '#ffc107'; $icon = '⚠️'; break;
            default: $color = '#198754'; $icon = '✅';
        }

        echo "<tr>
            <td>$time</td>
            <td>$icon $message</td>
            <td><span style='color: $color; font-weight: bold;'>$type</span></td>
            <td>$user_display</td>
        </tr>";
    }

    echo '</tbody></table>';

    // 📌 Page navigation
    if ($total_pages > 1) {
        echo '<div style="margin-top: 15px;">';
        for ($i = 1; $i <= $total_pages; $i++) {
            $active = ($i === $current_page) ? 'font-weight: bold; text-decoration: underline;' : '';
            echo '<a href="' . esc_url(add_query_arg(['paged' => $i, 'per_page' => $per_page])) . '" style="margin-right: 10px; ' . $active . '">Page ' . $i . '</a>';
        }
        echo '</div>';
    }

    // 🎛️ Select display per page
    echo '<form method="get" style="margin-top: 20px; display: inline-block;">';
    echo '<input type="hidden" name="page" value="ur-membership">';
    echo '<input type="hidden" name="tab" value="log">';
    echo 'Show <input type="number" name="per_page" value="' . esc_attr($per_page) . '" min="5" max="500" style="width:60px;"> entry by page ';
    echo '<button type="submit" style="margin-left:10px; background: #198754; color: white; border: none; padding: 8px 20px; border-radius: 4px; cursor: pointer;">🔄 Apply</button>';
    echo '</form>';
	
	// 🔁 Add a button below the table for export
	echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin-top: 20px; display: inline-block; margin-left: 20px;">';
    echo '<input type="hidden" name="action" value="export_ur_process_log">';
    echo '<button type="submit" style="background: #0d6efd; color: white; border: none; padding: 8px 20px; border-radius: 4px; cursor: pointer;">⬇️ Export CSV</button>';
    echo '</form>';
	
	// 🧹 Clear records button
    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" onsubmit="return confirm(\'⚠️ Do you want to delete all log messages?\');" style="margin-top: 20px; display: inline-block; margin-left: 20px;">';
    echo '<input type="hidden" name="action" value="clear_ur_process_log">';
    echo '<button type="submit" style="background: #dc3545; color: white; border: none; padding: 8px 20px; border-radius: 4px; cursor: pointer;">🧹 Clear records</button>';
    echo '</form>';
    echo '</div>'; // close padding
    echo '</div>'; // close frame

}

function add_to_ur_process_log($message, $type = 'info') {
    $log = get_option('ur_process_log', []);

    $log[] = [
        'timestamp' => current_time('timestamp'),
        'message'   => wp_strip_all_tags($message), // ⬅️ erases <strong>
        'type'      => $type,
        'user_id'   => get_current_user_id()
    ];

    update_option('ur_process_log', $log);
}

add_action('admin_post_clear_ur_process_log', function () {
    if (!current_user_can('manage_options')) {
        wp_die('⛔ You do not have sufficient permissions to perform this action.');
    }

    update_option('ur_process_log', []);
    set_process_message('🧹 The log process has been deleted.', 'warning');
    wp_redirect(admin_url('admin.php?page=ur-membership&tab=log'));
    exit;
});

add_action('admin_post_export_ur_process_log', function () {
    if (!current_user_can('manage_options')) {
        wp_die('⛔ You do not have sufficient permissions to perform this action.');
    }

    $log = get_option('ur_process_log', []);
    if (empty($log)) {
        wp_die('📭 No log records are available for export.');
    }

    // Preparing CSV
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="ur_process_log.csv"');

    $output = fopen('php://output', 'w');
    fputcsv($output, ['Time', 'Message', 'Type', 'User']);

    foreach ($log as $entry) {
        $time = date('d.m.Y. H:i:s', $entry['timestamp']);
        $message = $entry['message'];
        $type = $entry['type'];
        $user = get_userdata($entry['user_id'] ?? 0);
        $user_display = $user ? $user->user_login : '–';

        fputcsv($output, [$time, $message, $type, $user_display]);
    }

    fclose($output);
    exit;
});
